using System;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Threading;
using Microsoft.Win32;
using NDI.SLIKDA.Interop;
using CSWindowsService;


namespace CSWindowsService
{
	public class CServer
	{
		private ServerForm m_frm = null;
		private CSimulate m_SimObject = null;
		private Thread m_ServerThread = null;

		public CServer()
		{
		}

		public bool RegisterServer( string strServiceName )
		{
			if( m_frm != null )
				return false;

			bool bRet = true;
			m_frm = new ServerForm();
			try
			{
				m_frm.SLIKObject.RegisterServer();

				//
				//	After registering the OPC server functionality
				//	a linkage between this and the Windows Service
				//	functionality is required.  This linkage is made
				//	by adding a LocalService named value to the 
				//	AppID/{Application ID} subkey, which was created by 
				//	the above RegisterServer() call.
				//
				string strKey = "AppID\\" + m_frm.SLIKObject.AppID;
				RegistryKey regkey = Registry.ClassesRoot.OpenSubKey( strKey, true );
				regkey.SetValue( "LocalService", strServiceName );
				regkey.Close();
			}
			catch( System.Exception )
			{
				bRet = false;
			}

			return bRet;
		}

		public bool UnregisterServer()
		{
			if( m_frm != null )
				return false;

			bool bRet = true;
			m_frm = new ServerForm();
			try
			{
				m_frm.SLIKObject.UnregisterServer();
			}
			catch( System.Exception )
			{
				bRet = false;
			}

			return bRet;
		}

		public void StartServer()
		{
			if( m_ServerThread != null )
				return;

			m_ServerThread = new Thread(new ThreadStart(RunOPCServer));
			m_ServerThread.ApartmentState = ApartmentState.STA;
			m_ServerThread.Start();
		}

		public void StopServer()
		{
			if ( m_SimObject != null )
				m_SimObject.StopSimulation();
			if( m_frm != null )
				m_frm.Close();
		}

		//	The OPC server runs in it's own STA thread.  The SLIK-DA 
		//	Windows Form Control MUST execute in an STA context.
		[STAThread]
		private void RunOPCServer()
		{
			//	Create the hidden form housing the SLIK-DA Windows Form Control
			m_frm = new ServerForm();
			
			//	Initialize the OPC server namespace
			InitNamespace();

			//	Start the data simulator - runs in a separate MTA thread 
			m_SimObject = new CSimulate( m_frm.SLIKObject );
			m_SimObject.StartSimulation();

			//	Start the OPC server function
			m_frm.SLIKObject.StartServer();

			//	Start a message pump on the form - required for SLIK-DA
			//	event processing.  The event handlers are defined and
			//	and implemented in the ServerForm class.
			Application.Run( m_frm );
		}

		private void InitNamespace()
		{
			const short nTagsPerLeafFolder  = 10;
			int nAccess;
			short nIndex;
		
			try
			{
				ISLIKTags Tags = m_frm.SLIKObject.SLIKTags;
			
				//
				// Create a set of read-only tags that simulate data changes
				//
				nAccess = (int) AccessPermissionsEnum.sdaReadAccess;
				for( nIndex = 1; nIndex <= nTagsPerLeafFolder; nIndex++ )
				{
					Tags.Add( 
						"src.BOOL.tag" + nIndex.ToString("00"), 
						nAccess, 
						true, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);
				
					Tags.Add(
						"src.UI1.tag" + nIndex.ToString("00"), 
						nAccess, 
						(byte) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);
				
					Tags.Add(
						"src.I2.tag" + nIndex.ToString("00"), 
						nAccess, 
						(short) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"src.I4.tag" + nIndex.ToString("00"), 
						nAccess,
						(int) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"src.R4.tag" + nIndex.ToString("00"), 
						nAccess, 
						(float) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"src.R8.tag" + nIndex.ToString("00"), 
						nAccess, 
						(double) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"src.BSTR.tag" + nIndex.ToString("00"), 
						nAccess, 
						"" + nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"src.DATE.tag" + nIndex.ToString("00"), 
						nAccess, 
						System.DateTime.FromOADate(nIndex), 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"src.CY.tag" + nIndex.ToString("00"), 
						nAccess, 
						new System.Runtime.InteropServices.CurrencyWrapper(nIndex), 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);
				}

				//
				// Create a set of tags that any OPC client may change.
				//
				nAccess = (int) (AccessPermissionsEnum.sdaReadAccess | AccessPermissionsEnum.sdaWriteAccess);
				for( nIndex = 1; nIndex <= nTagsPerLeafFolder; nIndex++ )
				{
					Tags.Add(
						"trg.BOOL.tag" + nIndex.ToString("00"), 
						nAccess, 
						true, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"trg.UI1.tag" + nIndex.ToString("00"), 
						nAccess, 
						(byte) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"trg.I2.tag" + nIndex.ToString("00"), 
						nAccess, 
						(short) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"trg.I4.tag" + nIndex.ToString("00"), 
						nAccess, 
						(int) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"trg.R4.tag" + nIndex.ToString("00"), 
						nAccess, 
						(float) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"trg.R8.tag" + nIndex.ToString("00"), 
						nAccess, 
						(double) nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);
				
					Tags.Add(
						"trg.BSTR.tag" + nIndex.ToString("00"), 
						nAccess, 
						"" + nIndex, 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"trg.DATE.tag" + nIndex.ToString("00"), 
						nAccess, 
						System.DateTime.FromOADate(nIndex),
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);

					Tags.Add(
						"trg.CY.tag" + nIndex.ToString("00"), 
						nAccess, 
						new System.Runtime.InteropServices.CurrencyWrapper(nIndex), 
						(short) QualityStatusEnum.sdaGood, 
						DefaultValues.Add_InitialTimestamp, 
						DefaultValues.Add_AccessPaths
						);
				}
			}
			catch( System.Exception )
			{
			}
		}
	}
}
